'use client'

import { Button } from '@/components/ui/button'
import { Card } from '@/components/ui/card'
import { Github, Send, MessageCircle, Instagram, Phone, ExternalLink, Code2, Sparkles, Terminal } from 'lucide-react'
import { useState, useEffect } from 'react'

// Data
const terminalLines = [
  '> Initializing portfolio...',
  '> Loading skills...',
  '> Connecting to website...',
  '> Ready to view.'
] as const

const skills = [
  { year: '2022', technologies: ['HTML5', 'CSS3', 'JavaScript'], description: 'Memulai perjalanan coding dengan dasar web development' },
  { year: '2023', technologies: ['Python', 'JavaScript', 'React'], description: 'Mengembangkan skill dengan framework backend dan frontend' },
  { year: '2024', technologies: ['Next.js', 'TypeScript', 'Tailwind CSS'], description: 'Menguasai modern full-stack development' },
  { year: '2025', technologies: ['Full-Stack Development', 'API Integration', 'Cloud Services'], description: 'Membangun aplikasi production-ready' }
]

const projects = [
  {
    title: 'E-Commerce Platform',
    description: 'Full-featured online store with product catalog, shopping cart, and payment integration.',
    technologies: ['Next.js', 'Prisma', 'Stripe'],
    url: '#'
  },
  {
    title: 'Chat Application',
    description: 'Real-time messaging app with WebSocket, user authentication, and group chat features.',
    technologies: ['Socket.io', 'React', 'Node.js'],
    url: '#'
  },
  {
    title: 'Weather Dashboard',
    description: 'Weather app with location-based forecasts, hourly updates, and interactive maps.',
    technologies: ['OpenWeather API', 'TypeScript', 'Tailwind'],
    url: '#'
  },
  {
    title: 'Task Manager',
    description: 'Productivity app with drag-and-drop task organization, due dates, and progress tracking.',
    technologies: ['React', 'Zustand', 'SQLite'],
    url: '#'
  },
  {
    title: 'API Integration Tool',
    description: 'Tool for testing and documenting REST APIs with request builder and response viewer.',
    technologies: ['Next.js', 'Axios', 'Monaco'],
    url: '#'
  },
  {
    title: 'Portfolio Website',
    description: 'Modern portfolio with animated terminal, project showcase, and responsive design.',
    technologies: ['Next.js 15', 'Tailwind CSS', 'shadcn/ui'],
    url: '#'
  }
]

const contacts = [
  { platform: 'WhatsApp', handle: '+62 821 9735 0500', url: 'https://wa.me/6282197350500', icon: Phone, color: 'text-emerald-400' },
  { platform: 'TikTok', handle: '@jayellstr', url: 'https://tiktok.com/@jayellstr', icon: MessageCircle, color: 'text-white' },
  { platform: 'Instagram', handle: '@senazqt', url: 'https://Instagram.com/senazqt', icon: Instagram, color: 'text-pink-400' },
  { platform: 'GitHub', handle: '@jaynatusvincere', url: 'https://github.com/jaynatusvincere', icon: Github, color: 'text-white' },
  { platform: 'Telegram', handle: '@holaajay', url: 'https://t.me/holaajay', icon: Send, color: 'text-blue-400' },
  { platform: 'Saluran WhatsApp', handle: 'Join Channel', url: 'https://whatsapp.com/channel/0029Vb6s2vYDeON4gU4i4N2J', icon: MessageCircle, color: 'text-emerald-400' }
]

// Components
function SectionTitle({ title, accent = 'blue', icon: Icon }: { title: string; accent?: 'blue' | 'cyan'; icon?: any }) {
  const colorClass = accent === 'blue' ? 'text-blue-400' : 'text-cyan-400'
  return (
    <div className="text-center mb-12 md:mb-16">
      <div className="inline-flex items-center gap-3 mb-4">
        {Icon && <Icon className={`w-6 h-6 ${colorClass}`} />}
        <span className={`h-px w-12 ${colorClass} bg-gradient-to-r ${accent === 'blue' ? 'from-blue-400 to-transparent' : 'from-cyan-400 to-transparent'}`} />
      </div>
      <h2 className="text-3xl md:text-5xl font-bold text-white tracking-tight">
        <span className={colorClass}>&lt;</span>
        {title}
        <span className={colorClass}>&gt;</span>
      </h2>
    </div>
  )
}

function ProjectCard({ title, description, technologies, url }: typeof projects[0]) {
  return (
    <Card className="group relative bg-gradient-to-br from-slate-900/95 via-blue-950/50 to-slate-900/95 border border-blue-500/20 backdrop-blur-sm overflow-hidden transition-all duration-500 hover:border-cyan-500/40 hover:shadow-2xl hover:shadow-cyan-500/10 hover:-translate-y-1">
      {/* Glow Effect */}
      <div className="absolute inset-0 bg-gradient-to-r from-blue-500/5 via-cyan-500/5 to-blue-500/5 opacity-0 group-hover:opacity-100 transition-opacity duration-500" />

      <div className="relative p-6 md:p-8 flex flex-col h-full">
        <div className="flex-grow">
          <div className="flex items-start justify-between mb-4">
            <h3 className="text-xl md:text-2xl font-bold bg-gradient-to-r from-cyan-400 to-blue-400 bg-clip-text text-transparent leading-tight">
              {title}
            </h3>
            <ExternalLink className="w-5 h-5 text-blue-400 opacity-0 group-hover:opacity-100 transition-opacity duration-300 flex-shrink-0 mt-1" />
          </div>
          <p className="text-slate-300 text-sm md:text-base leading-relaxed mb-6">
            {description}
          </p>
          <div className="flex flex-wrap gap-2">
            {technologies.map((tech) => (
              <span
                key={tech}
                className="px-3 py-1.5 bg-blue-500/10 border border-blue-500/20 rounded-md text-blue-300 text-xs font-mono tracking-wide hover:bg-blue-500/20 transition-colors duration-300"
              >
                {tech}
              </span>
            ))}
          </div>
        </div>
        <Button asChild size="sm" className="mt-6 bg-gradient-to-r from-cyan-500 to-blue-600 hover:from-cyan-400 hover:to-blue-500 text-white text-xs font-medium px-6 py-2.5 w-full border border-cyan-400/20">
          <a href={url} target="_blank" rel="noopener noreferrer" className="flex items-center justify-center gap-2">
            View Project
            <ExternalLink className="w-3.5 h-3.5" />
          </a>
        </Button>
      </div>
    </Card>
  )
}

function SkillTimelineItem({ year, technologies, description }: typeof skills[0]) {
  return (
    <div className="group relative pl-10 md:pl-14">
      {/* Timeline Line */}
      <div className="absolute left-0 top-0 bottom-0 w-px bg-gradient-to-b from-blue-500/30 via-blue-500/50 to-blue-500/30" />

      {/* Year Badge */}
      <div className="absolute -left-4 md:-left-5 top-0 bg-gradient-to-r from-blue-600 to-blue-500 text-white px-4 py-2 rounded-full font-bold text-sm md:text-base shadow-lg shadow-blue-500/30 ring-2 ring-blue-400/20 group-hover:ring-blue-400/40 transition-all duration-300 group-hover:scale-105 z-10">
        {year}
      </div>

      {/* Skills Card */}
      <Card className="ml-0 md:ml-4 mt-6 bg-gradient-to-r from-slate-900/80 to-blue-950/40 border border-blue-500/15 backdrop-blur-sm p-6 md:p-8 transition-all duration-300 group-hover:border-blue-500/30 group-hover:shadow-xl group-hover:shadow-blue-500/10 group-hover:translate-x-1">
        <div className="mb-5">
          <p className="text-cyan-400/90 text-sm font-medium mb-2 leading-relaxed">
            {description}
          </p>
        </div>
        <div className="flex flex-wrap gap-2.5">
          {technologies.map((tech) => (
            <span
              key={tech}
              className="px-3 py-1.5 bg-blue-500/10 border border-blue-500/20 rounded-md text-blue-300 font-mono text-xs md:text-sm hover:bg-blue-500/20 hover:border-blue-400/40 transition-all duration-300"
            >
              {tech}
            </span>
          ))}
        </div>
      </Card>
    </div>
  )
}

function ContactCard({ platform, handle, url, icon: Icon, color }: typeof contacts[0]) {
  return (
    <a href={url} target="_blank" rel="noopener noreferrer" className="group">
      <Card className="relative bg-gradient-to-br from-slate-900/90 to-blue-950/30 border border-blue-500/15 backdrop-blur-sm p-6 md:p-8 transition-all duration-300 hover:border-cyan-500/40 hover:shadow-xl hover:shadow-cyan-500/10 hover:-translate-y-1 h-full overflow-hidden">
        {/* Glow Effect on Hover */}
        <div className="absolute inset-0 bg-gradient-to-br from-cyan-500/5 via-blue-500/5 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-500" />

        <div className="relative">
          <div className={`w-14 h-14 rounded-xl bg-gradient-to-br ${color.replace('text-', 'from-').replace('-400', '-500/20')} ${color.replace('text-', 'to-').replace('-400', '-600/20')} flex items-center justify-center mb-5 ring-1 ${color.replace('text-', 'ring-').replace('-400', '/20')} group-hover:scale-110 group-hover:ring-opacity-40 transition-all duration-300`}>
            <Icon className={`w-7 h-7 ${color}`} />
          </div>
          <h3 className="text-lg font-semibold text-white mb-2 tracking-tight">{platform}</h3>
          <p className="text-slate-400 text-sm font-mono leading-relaxed break-all group-hover:text-slate-300 transition-colors duration-300">
            {handle}
          </p>
        </div>
      </Card>
    </a>
  )
}

// Main Component
export default function Home() {
  const [typingText, setTypingText] = useState('')
  const [currentLine, setCurrentLine] = useState(0)

  useEffect(() => {
    let timeout: NodeJS.Timeout
    if (currentLine < terminalLines.length) {
      const fullText = terminalLines[currentLine]
      let index = 0
      const typeInterval = setInterval(() => {
        if (index < fullText.length) {
          setTypingText(prev => {
            const newText = terminalLines.slice(0, currentLine).join('\n') + '\n' + fullText.slice(0, index + 1)
            return newText
          })
          index++
        } else {
          clearInterval(typeInterval)
          timeout = setTimeout(() => {
            setCurrentLine(prev => prev + 1)
          }, 400)
        }
      }, 40)
    }
    return () => {
      clearInterval(timeout as unknown as NodeJS.Timeout)
    }
  }, [currentLine])

  return (
    <div className="min-h-screen bg-[#0a0f1c] text-white flex flex-col">
      {/* Animated Background */}
      <div className="fixed inset-0 overflow-hidden pointer-events-none">
        {/* Gradient Blobs */}
        <div className="absolute top-0 left-1/4 w-96 h-96 bg-blue-500/5 rounded-full blur-3xl animate-pulse" style={{ animationDuration: '8s' }} />
        <div className="absolute top-1/3 right-1/4 w-[500px] h-[500px] bg-cyan-500/5 rounded-full blur-3xl animate-pulse" style={{ animationDelay: '2s', animationDuration: '10s' }} />
        <div className="absolute bottom-1/4 left-1/3 w-[400px] h-[400px] bg-blue-600/5 rounded-full blur-3xl animate-pulse" style={{ animationDelay: '4s', animationDuration: '12s' }} />

        {/* Grid Pattern */}
        <div className="absolute inset-0 bg-[linear-gradient(rgba(59,130,246,0.02)_1px,transparent_1px),linear-gradient(90deg,rgba(59,130,246,0.02)_1px,transparent_1px)] bg-[size:60px_60px] [mask-image:radial-gradient(ellipse_at_center,black_40%,transparent_70%)]" />
      </div>

      {/* Main Content */}
      <main className="relative z-10 flex-grow">
        {/* Hero Section */}
        <section className="min-h-screen flex items-center justify-center px-4 py-20 md:py-24">
          <div className="max-w-5xl w-full">
            <Card className="bg-slate-950/95 border border-blue-500/25 backdrop-blur-xl overflow-hidden shadow-2xl shadow-blue-500/10">
              {/* Terminal Header */}
              <div className="bg-gradient-to-r from-slate-900 to-blue-950/80 px-5 py-3.5 flex items-center gap-3 border-b border-blue-500/20">
                <div className="flex gap-2.5">
                  <div className="w-3 h-3 rounded-full bg-red-500/80 hover:bg-red-500 transition-colors duration-200" />
                  <div className="w-3 h-3 rounded-full bg-yellow-500/80 hover:bg-yellow-500 transition-colors duration-200" />
                  <div className="w-3 h-3 rounded-full bg-green-500/80 hover:bg-green-500 transition-colors duration-200" />
                </div>
                <div className="flex items-center gap-2 ml-2 flex-1">
                  <Terminal className="w-4 h-4 text-blue-400/60" />
                  <span className="text-xs font-mono text-slate-400 tracking-wide">jay@portfolio:~</span>
                </div>
              </div>

              {/* Terminal Content */}
              <div className="p-8 md:p-12 font-mono text-sm md:text-base">
                <pre className="text-blue-300/90 whitespace-pre-wrap overflow-x-auto leading-relaxed mb-6">
                  <code>{typingText}</code>
                </pre>
                {currentLine >= terminalLines.length && (
                  <div className="animate-in fade-in slide-in-from-bottom-4 duration-700">
                    <div className="flex items-center gap-3 mb-4">
                      <Sparkles className="w-6 h-6 text-cyan-400 animate-pulse" />
                      <span className="text-xs text-slate-400 uppercase tracking-widest">Welcome</span>
                    </div>
                    <h1 className="text-4xl md:text-6xl lg:text-7xl font-bold bg-gradient-to-r from-blue-400 via-cyan-400 to-blue-500 bg-clip-text text-transparent mb-6 leading-tight tracking-tight">
                      JAY NATUS VINCERE
                    </h1>
                    <p className="text-lg md:text-xl text-slate-300 max-w-2xl leading-relaxed">
                      <span className="text-cyan-400 font-medium">Full-Stack Developer</span>
                      {' '}— Crafting digital experiences with code and creativity.
                    </p>
                  </div>
                )}
              </div>
            </Card>
          </div>
        </section>

        {/* Divider */}
        <div className="w-full h-px bg-gradient-to-r from-transparent via-blue-500/30 to-transparent my-16 md:my-24" />

        {/* Skills Timeline Section */}
        <section className="px-4 py-16 md:py-32">
          <div className="max-w-4xl mx-auto">
            <SectionTitle title="Coding Journey" accent="blue" icon={Code2} />
            <div className="space-y-16 md:space-y-20">
              {skills.map((skill) => (
                <SkillTimelineItem key={skill.year} {...skill} />
              ))}
            </div>
          </div>
        </section>

        {/* Divider */}
        <div className="w-full h-px bg-gradient-to-r from-transparent via-cyan-500/30 to-transparent my-16 md:my-24" />

        {/* Projects Section */}
        <section className="px-4 py-16 md:py-24">
          <div className="max-w-6xl mx-auto">
            <SectionTitle title="My Projects" accent="cyan" icon={Sparkles} />
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 md:gap-8">
              {projects.map((project, index) => (
                <ProjectCard key={index} {...project} />
              ))}
            </div>
          </div>
        </section>

        {/* Divider */}
        <div className="w-full h-px bg-gradient-to-r from-transparent via-blue-500/30 to-transparent my-16 md:my-24" />

        {/* Contact Section */}
        <section className="px-4 py-16 md:py-24 pb-32">
          <div className="max-w-6xl mx-auto">
            <SectionTitle title="Get In Touch" accent="blue" icon={MessageCircle} />
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-5 md:gap-6">
              {contacts.map((contact, index) => (
                <ContactCard key={index} {...contact} />
              ))}
            </div>
          </div>
        </section>
      </main>

      {/* Sticky Footer */}
      <footer className="relative z-10 mt-auto bg-slate-950/90 border-t border-blue-500/15 backdrop-blur-xl">
        <div className="max-w-6xl mx-auto px-4 py-6 md:py-8">
          <div className="flex flex-col md:flex-row items-center justify-between gap-4">
            <p className="text-slate-400 text-sm text-center md:text-left">
              © Jay Portfolio - 2025
            </p>
            <p className="text-slate-500 text-xs font-mono flex items-center gap-2">
              <Code2 className="w-4 h-4 text-blue-400" />
              Built with{' '}
              <span className="text-blue-400">Next.js</span>
              <span className="text-slate-600">,</span>{' '}
              <span className="text-cyan-400">TypeScript</span>
              <span className="text-slate-600">,</span>{' '}
              <span className="text-white">&</span>{' '}
              <span className="text-slate-300">Tailwind CSS</span>
            </p>
          </div>
        </div>
      </footer>
    </div>
  )
}
